<?php

namespace App\Http\Controllers\Admin;

use App\Models\ActivityLog;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;

class ActivityLogController extends Controller
{
    /**
     * Display activity logs
     */
    public function index(Request $request)
    {
        $query = ActivityLog::with('user');

        // Filter by search (user name)
        if ($request->filled('search')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->search . '%');
            });
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->where('role', $request->role);
        }

        // Filter by module
        if ($request->filled('module')) {
            $query->where('module', $request->module);
        }

        // Filter by action
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Get paginated logs
        $logs = $query->latest()->paginate(20);

        // Calculate stats
        $stats = [
            'total' => ActivityLog::count(),
            'create' => ActivityLog::where('action', 'create')->count(),
            'update' => ActivityLog::where('action', 'update')->count(),
            'delete' => ActivityLog::where('action', 'delete')->count(),
        ];

        return view('admin.activity-log.index', compact('logs', 'stats'));
    }

    /**
     * Clear old logs (optional - untuk maintenance)
     */
    public function clearOldLogs()
    {
        // Hapus log lebih dari 90 hari
        $deleted = ActivityLog::where('created_at', '<', now()->subDays(90))->delete();

        return back()->with('success', "Berhasil menghapus {$deleted} log lama");
    }
}
